<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * @package DKS
 * @version 1.1.8
 */
/*
Plugin Name: DKS Client
Plugin URI: https://www.epc.de
Description: DKS Client Plug-in for Wordpress
Author: EPC
Version: v2.1.35
Author URI: https://www.epc.de
Text Domain: dks
Domain Path: /languages
*/

class DudenKorrekturTinyMCEPlugin
{
	private $settings;
	protected $options;
	/**
	 * Constructor. Called when the plugin is initialised.
	 */
	function __construct()
	{
		error_log(__CLASS__."::".__FUNCTION__." (Line ".__LINE__.") ".is_admin());

		if (is_admin()) {
			require_once dirname( __FILE__ ) . '/vendor/class.settings-api.php';
			require_once dirname( __FILE__ ) . '/includes/dks-settings.php';
			add_action('plugins_loaded', array($this, 'load_l10n'));
			add_action('wp_print_scripts',array($this, 'print_dkce_settings'));
			add_action('admin_enqueue_scripts', array($this, 'setup_plugin'));
			add_action('init', array($this, 'init_plugin'));

			$this->settings = new DKS_Settings(
				__( 'DKS Client', 'dks' ),
				__( 'DKS Client', 'dks' ),
				'dks-settings'
			);

			$this->options = array_merge(
				$this->getOption(DKS_Settings::OPTION_NAME),
				$this->getOption(DKS_Settings::SETTINGS_NAME),
				$this->getOption(DKS_Settings::ADVANCED),
			);
		}
	}

	function getOption($optionId) {
		$result = get_option($optionId) ?? '';
		if (!(is_array($result))) return [];
		return $result;
	}

	function getBoolOption($optionId) {
    	$result = $this->options[$optionId] ?? 'off';
		return $result === 'on';
    }

	function is_gutenberg_editor()
	{
		if (function_exists('is_gutenberg_page') && is_gutenberg_page())
			return true;

		$current_screen = get_current_screen();
		if (method_exists($current_screen, 'is_block_editor') && $current_screen->is_block_editor())
			return true;

		return false;
	}

	function getPropertySets() {
		$propertySets = [];
		if (is_array($this->options['propertySets'] ?? NULL)) {
    		$propertySets = array_values($this->options['propertySets']);
		}
		return $propertySets;
	}

	function getDictionaries() {
		$dictionaries = [];
		if (is_array($this->options['dictionaries'] ?? NULL)) {
			$dictionaries = array_values($this->options['dictionaries']);
		}
		return $dictionaries;
	}

	function print_dkce_settings() {
		$json = $this->options['json'] ?? '{}';
		$json = json_decode($json, true);

		$dkceSettings = [
			'apiEndpoint' => plugin_dir_url(__FILE__) . "dks-api-proxy.php?p=/api",
			'path' => plugin_dir_url(__FILE__),
			'spellcheckingLevel' => 3,
			'overlayMode' => false,
			'messageLevel' => 3,
			'propertySets' => $this->getPropertySets(),
			'dictionaries' => $this->getDictionaries(),
			'openContextMenuOn' => $this->options['openContextMenuOn'] ?? 'mouseOver',
			'debug' => $this->getBoolOption('debug'),
			'debugCanvas' => $this->getBoolOption('debugCanvas')
		];

		if (is_array($json))
			$dkceSettings = array_merge($dkceSettings, $json);
		?>
		<script type="text/javascript">
		    var DKCE_SETTINGS = <?php echo json_encode($dkceSettings) ?>;
		</script>
		<?php
	}

	function init_plugin()  {
	}

	/**
	 * Check if the current user can edit Posts or Pages, and is using the Visual Editor
	 * If so, add some filters so we can register our plugin
	 */
	function setup_plugin()
	{
		error_log(__CLASS__ . "::" . __FUNCTION__ . " (Line " . __LINE__ . ")");

		// Check if the logged in WordPress User can edit Posts or Pages
		// If not, don't register our TinyMCE plugin
		error_log(__CLASS__ . "::" . __FUNCTION__ . " (Line " . __LINE__ . ") " . current_user_can('edit_posts') . ", " . current_user_can('edit_pages') . ", " . get_user_option('rich_editing'));
		if (!current_user_can('edit_posts') && !current_user_can('edit_pages')) {
			return;
		}

		wp_enqueue_style(
			'dks-plugin',
			plugin_dir_url(__FILE__) . 'dks-plugin.css'
		);

		wp_enqueue_script(
			'dks-plugin',
			plugins_url('dkce.js', __FILE__),
			array(),
			'1.0',
			true
		);

		wp_localize_script('dks-plugin', 'dksl18n', array(
			'dksname' => __("DKS Client", "dks"),
			'spellcheck' => __("Spell check", "dks")
		));


		if ($this->is_gutenberg_editor()) {
			wp_enqueue_script(
				'dks-plugin-gutenberg',
				plugins_url('gutenberg.js', __FILE__),
				['wp-blocks', 'wp-components', 'wp-element'],
				'1.0',
				true
			);
		} else {
			// Setup some filters
			add_filter('mce_external_plugins', array(&$this, 'add_tinymce_plugin'));
			add_filter('mce_buttons', array(&$this, 'add_tinymce_toolbar_button'));
		}
	}

	/**
	 * Adds a TinyMCE plugin compatible JS file to the TinyMCE / Visual Editor instance
	 *
	 * @param array $plugin_array Array of registered TinyMCE Plugins
	 * @return array Modified array of registered TinyMCE Plugins
	 */
	function add_tinymce_plugin($plugin_array)
	{

		error_log(__CLASS__."::".__FUNCTION__." (Line ".__LINE__.")");

		$plugin_array['dks'] = plugin_dir_url(__FILE__) . 'tinymce-plugin.js';
		return $plugin_array;
	}

	/**
	 * Adds a button to the TinyMCE / Visual Editor which the user can click
	 * to insert a link with a custom CSS class.
	 *
	 * @param array $buttons Array of registered TinyMCE Buttons
	 * @return array Modified array of registered TinyMCE Buttons
	 */
	function add_tinymce_toolbar_button($buttons)
	{

		array_push($buttons, '|', 'dks');
		return $buttons;
	}

	function load_l10n() {
		$plugin_rel_path = basename(dirname(__FILE__)) . '/languages';
		$res = load_plugin_textdomain('dks', false, $plugin_rel_path);
	}
}

$dudenKorrekturTinyMCEPlugin = new DudenKorrekturTinyMCEPlugin;