<?php

if ( ! defined( 'ABSPATH' ) ) {
	error_log("EXITING");
	exit; // Exit if accessed directly.
}

/**
 * WordPress settings API class
 *
 */
if ( ! class_exists( 'WeDevs_Settings_API_Test' ) ) {
	class DKS_Settings {
		const OPTION_NAME = 'dks';
		const SETTINGS_NAME = 'dks-settings';
		const ADVANCED = 'dks-advanced';


		/**
		 * @var WeDevs_Settings_API
		 */
		private $settings_api;
		private $page_title;
		private $menu_title;
		private $menu_slug;

		function __construct( $page_title, $menu_title, $menu_slug ) {
			$this->settings_api = new WeDevs_Settings_API;
			$this->menu_title   = $menu_title;
			$this->menu_slug    = $menu_slug;
			$this->page_title   = $page_title;
			add_action( 'admin_init', array( $this, 'admin_init' ) );
			add_action( 'admin_menu', array( $this, 'admin_menu' ) );
		}

		/**
		 * Helper method, so we can make direct requests to the API without going
		 * through the proxy script.
		 *
		 * TODO(jochen) It would be nice if we have like a small and simple PHP Client Class
		 * we can include from within this plugin and the api_proxy script.
		 */
		function api($method) {
			$options = !empty(get_option('dks')) ? get_option('dks') : array();
			if (!array_key_exists('server', $options)) return [];

			$DKS_URL = rtrim($options['server'], DIRECTORY_SEPARATOR);
			$DKS_API_TOKEN = $options['apikey'] ?? NULL;
			$headers = array();
			if (isset($DKS_API_TOKEN)) {
				array_push($headers, "Authorization: Bearer " . $DKS_API_TOKEN);
			}
			$ch = curl_init($DKS_URL . '/api/' . $method);
			curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
			curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'GET');
			curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);

			$result = curl_exec($ch);
			if ($result === '') return [];
			if (curl_errno($ch)) return [];
			curl_close($ch);

			return $result !== false ? json_decode($result, true) : [];
		}

		function admin_init() {
			$this->propsets = $this->api('propset');
			$this->dictionaries = $this->api('dictionaries');
			$this->version  = $this->api('version')['Server version'] ?? NULL;

			//set the settings
			$this->settings_api->set_sections( $this->get_settings_sections() );
			$this->settings_api->set_fields( $this->get_settings_fields() );

			//initialize settings
			$this->settings_api->admin_init();
//			$this->set_default_settings();

			// Initialize Script for dynamic DKS State determination
			wp_enqueue_script('determine-dks-state', plugins_url('js/determine-dks-state.js', __FILE__), [], '1.0.0', ['strategy' => 'defer']);

			$proxyScriptURL = substr(dirname(__FILE__), 0, -strlen('includes')) . '/dks-api-proxy.php';
			$pluginDirectory = plugins_url(basename($proxyScriptURL), $proxyScriptURL);

			wp_localize_script('determine-dks-state', 'dks_state', array(
				'plugin_directory' => $pluginDirectory,
				'i18n' => array(

					'dksStateOK'   => sprintf(__('The connection to the DKS v%s is established.', 'dks'), $this->version ?? '0.0.0'),
					'dksState401'  => __('Authentication to DKS failed (error code: 401)', 'dks'),
					'dksStateFail' => __('Connection to DKS failed.', 'dks'),
					'dksStatePropsetsNotLoaded' => __('Could not load Correction Profiles.', 'dks')
				)
			));

			// Add CSS styles
			wp_enqueue_style('dks-admin-styles', plugins_url('css/dks-admin.css', __FILE__));
		}

		public function set_default_settings( $settings ) {
			$this->settings_api->set_fields( $settings );
		}

		function admin_menu() {
			add_options_page( $this->page_title, $this->menu_title, 'delete_posts', $this->menu_slug, array(
				$this,
				'plugin_page'
			) );
		}

		function get_settings_sections() {
			$sections = [];
			$sections[] = [
				'id'    => DKS_Settings::OPTION_NAME,
				'title' => __( 'Connection', 'dks' )
			];

			if (isset($this->version)) {
				$sections[] = [
					'id'    => DKS_Settings::SETTINGS_NAME,
					'title' => __( 'Settings', 'dks' )
				];
			}

			$sections[] = [
				'id' => DKS_Settings::ADVANCED,
				'title' => __('Advanced', 'dks'),
			];

			return $sections;
		}

		/**
		 * Returns all the settings fields
		 *
		 * @return array settings fields
		 */
		function get_settings_fields() {
			$propsets = array();
			foreach($this->propsets as $item) {
				$propsets[$item['name']] = $item['name'];
			}

			$dictionaries = array();
			foreach ($this->dictionaries as $item) {
				$dictionaries[$item] = $item;
			}

			$settings_fields = [
				DKS_Settings::OPTION_NAME => [
					[
						'name'              => 'server',
						'label'             => __( 'DKS Host', 'dks' ),
						'desc'              => __( 'URL of DKS', 'dks' ),
						'type'              => 'text',
						'default'           => 'http://dks',
						'sanitize_callback' => 'sanitize_text_field',
					],
					[
						'name'              => 'apikey',
						'label'             => __( 'API Key', 'dks' ),
						'desc'              => __( 'API Key for access to secured DKS servers', 'dks' ),
						'type'              => 'text',
						'default'           => '',
						'sanitize_callback' => 'sanitize_text_field'
					]
				],
				DKS_Settings::SETTINGS_NAME => [
					[
						'name' => 'propertySets',
						'label' => __( 'Correction profiles', 'dks' ),
						'desc' => '',
						'type' => 'multicheck',
						'default' => ['base' => 'base'],
						'options' => $propsets
					],
					[
						'name' => 'dictionaries',
						'label' => __( 'Dictionaries', 'dks' ),
						'desc' => '',
						'type' => 'multicheck',
						'default' => [],
						'options' => $dictionaries
					],
					[
						'name' => 'openContextMenuOn',
						'label' => __('Open Contexte Menu on', 'dks'),
						'desc' => '',
						'type' => 'select',
						'default' => 'mouseOver',
						'options' => [
							'mouseOver'  => __('Mouse Over', 'dks'),
							'leftClick'	 => __('Left Mouse Click', 'dks'),
							'rightClick' => __('Right Mouse Click', 'dks')
						]
					]

				],
				DKS_Settings::ADVANCED => [
					[
						'name' => 'debug',
						'label' => __('Debug', 'dks'),
						'desc' => '',
						'default' => 'off',
						'type' => 'checkbox'
					],
					[
						'name' => 'debugCanvas',
						'label' => __('Debug Rendering', 'dks'),
						'desc' => '',
						'default' => 'off',
						'type' => 'checkbox'
					],
					[
						'name' => 'json',
						'label' => __('JSON Settings', 'dks'),
						'desc' => '',
						'type' => 'textarea',
						'default' => '{}' ?? json_encode([
							'detectManualHyphenation' => false,
							'hyphenation' => true,
							'hyphenationStandard' => "combined",
							'hyphenationInStem' => true,
							'hyphenationUnaest' => false,
							'ignoreAttributes' => [],
							'layerAttributes' => [],
							'useSingleWordModeOnLayers' => true,
							'ignoreClasses' => [],
							'ignoreCharacters' => '',
							'checkLanguages' => [],
							'correctionProposals' => true,
							'singleWordMode' => false,
							'styleForeign' => false,
							'styleOld' => false,
							'styleRegional' => false,
							'styleColloquial' => false,
							'styleFiller' => false,
							'markupMode' => "text",
							'messageLanguage' => "de-DE",
							'textLanguage' => "de-DE",
							'messageLevel' => 3,
							'spellcheckingLevel' => 1,
							'orthographyStandard' => "duden",
							'proposalDictionary' => "Proposals",
							'colors' => [
								'spellingError' => '#faa6ae',
								'gramarError' => '#abe89e',
								'rejectedWord' => '#abe89e',
								'styleError' => '#ffd7a8',
								'glossaryInfo' => '#ffff00'
							]
						], JSON_PRETTY_PRINT)
					]
				]
			];

			return apply_filters( 'wsc_admin_fields', $settings_fields );
		}

		function plugin_page() {
			echo '<div class="wrap">';
			echo "<h1>$this->page_title</h1>";
			echo "<div id='dks-status-area'></div>";
			$this->settings_api->show_navigation();
			$this->settings_api->show_forms();
			echo '</div>';
		}

		/**
		 * Get all the pages
		 *
		 * @return array page names with key value pairs
		 */
		function get_pages() {
			$pages         = get_pages();
			$pages_options = array();
			if ( $pages ) {
				foreach ( $pages as $page ) {
					$pages_options[ $page->ID ] = $page->post_title;
				}
			}

			return $pages_options;
		}

	}

}