(function() {
    tinymce.PluginManager.requireLangPack('duden');
    /**
     * Storage class to save user settings in browser local cache,
     * each tinymce instance has a separate storage
     */
    class Storage {
        constructor(editorId) {
            this.id = window.location.href + "_" + editorId;
        }

        getInstanceKey(key) {
            return this.id + "_" + key;
        }

        setLanguage(lang) {
            localStorage.setItem(this.getInstanceKey('language'), lang);
        }

        getLanguage(defaultValue) {
            return localStorage.getItem(this.getInstanceKey('language')) || defaultValue;
        }

        setSpellcheckingLevel(level) {
            localStorage.setItem(this.getInstanceKey('spellchecking-level'), level);
        }

        getSpellcheckingLevel(defaultValue) {
            return parseInt(localStorage.getItem(this.getInstanceKey('spellchecking-level'))) || defaultValue;
        }

        setOrthographyStandard(ortography) {
            localStorage.setItem(this.getInstanceKey('orthography-standard'), ortography);
        }

        getOrthographyStandard(defaultValue) {
            return localStorage.getItem(this.getInstanceKey('orthography-standard')) || defaultValue;
        }

        setActive(active) {
            localStorage.setItem(this.getInstanceKey('spellcheck_active'), JSON.stringify(!!active));
        }

        getActive() {
            var result = localStorage.getItem(this.getInstanceKey('spellcheck_active'));
            if (result == null) {
                return true;
            }
            result = JSON.parse(result);

            return result;
        }
    }

    /**
     * Wrapper object that contains the TinyMCE, Storage and DKCE API instance that
     * are belonging together
     */
    class DKCEPlugin {
        constructor(ed, pluginUrl) {
            this.id = ed.editorId || ed.id;
            this.editor = ed;
            this.storage = new Storage(this.id);
            this.pluginUrl = pluginUrl;
            this.locked = false;
            this.api = null;

            if (this.editor && typeof this.editor.on === 'function') {
                this.editor.on('remove', (e) => {
                    this.api?.activate(false, false);
                });
            }
        }

        /**
         * Initialize DKCE Api
         * @param {*} onSuccess optional Callback that gets called when API is initialized successfully
         */
        init(onSuccess) {
            var rootNode = this.editor.getBody();
            var settings = this.getPluginConfiguration();

            // check storage for text language and overwrite configuration
            var storagedSettings = {
                textLanguage: this.storage.getLanguage(null),
                spellcheckingLevel: this.storage.getSpellcheckingLevel(null),
                orthographyStandard: this.storage.getOrthographyStandard(null)
            };

            if (storagedSettings.textLanguage) {
                settings.textLanguage = storagedSettings.textLanguage;
            }

            if (storagedSettings.spellcheckingLevel) {
                settings.spellcheckingLevel = storagedSettings.spellcheckingLevel;
            }

            if (storagedSettings.orthographyStandard) {
                settings.orthographyStandard = storagedSettings.orthographyStandard;
            }

            const toolbar = JSON.stringify(this.editor.options?.get('toolbar') ?? this.editor.settings?.toolbar)?.toLowerCase() ?? '';
            const toolbarContainsDudenAI = toolbar.indexOf('duden_ai') > -1;

            if (settings.ai != toolbarContainsDudenAI) {
                console.warn("[Duden TinyMCE Plugin] You want to use the AI assistants of the DUDEN plugin. You need to add the button `duden_ai` to the toolbar settings and set the `ai` property of `duden_settings` to true as well.");
            }

            var scriptLoader = new tinymce.dom.ScriptLoader();
            scriptLoader.add(settings.path + '/js/dkce.js');

            var onScriptSuccess = () => {
                DKCE.init(rootNode, settings, (api) => {
                    this.api = api;
                    this.api.activate(this.storage.getActive(), false);
                    if (onSuccess) {
                        onSuccess();
                    }
                }, (e) => {
                    console.error(e);
                });
            };

            var onScriptFail = (e) => {
                console.error(e);
            };

            if (tinyMCE.majorVersion === '6' || tinyMCE.majorVersion === '7') {
                scriptLoader.loadQueue().then(onScriptSuccess).catch(onScriptFail);
            } else {
                scriptLoader.loadQueue(onScriptSuccess, onScriptFail);
            }
        }

        /**
         * Returns a JS object that contains all settings (uses fallbacks if possible)
         *
         * @param {*} settings
         * @param {*} tinymceSettings
         * @returns {Object} settings object
         */
        getPluginConfiguration() {
            var basePath = (new URL(tinyMCE.baseURI.toAbsolute('plugins/duden'))).pathname;
            var defaultApiEndpoint = (new URL(tinyMCE.baseURI.toAbsolute('plugins/duden/php/duden-correction-server-connector.php'))).pathname;

            var settings = {
                path: basePath,
                spellcheckingLevel: 3,
                overlayMode: true,
                messageLevel: 3,
                activated: true,
                zIndex: 1
            };

            var fallbackParams = [
                'duden_markup_element',
                'duden_default_text_language',
                'duden_proxy_script_url'
            ];

            // Use new TinyMCE 6 Options API
            if (typeof(this.editor.options) === 'object') {
                for (var key in settings) {
                    var value = settings[key];
                    this.editor.options.register(key, {
                        processor: typeof (value),
                        default: value
                    })

                    var configValue = this.editor.options.get(key);
                    if (typeof (configValue) !== 'undefined') settings[key] = configValue;
                }

                for (var key in fallbackParams) {
                    var value = settings[key];
                    this.editor.options.register(key, {
                        processor: 'string',
                        default: undefined
                    })

                    var configValue = this.editor.options.get(key);
                    if (typeof (configValue) !== 'undefined') settings[key] = configValue;
                }

                this.editor.options.register('duden_settings', {
                    processor: 'object',
                    default: {}
                })

                settings = Object.assign(settings, this.editor.options.get('duden_settings'));
            } else {
                var fallbackSettings = Object.fromEntries(Object.entries(this.editor.settings).filter(([key]) => fallbackParams.indexOf(key) > -1));

                settings = Object.assign(settings, fallbackSettings);
                settings = Object.assign(settings, this.editor.settings.duden_settings);
            }

            // convert old properties for better backward compatibility
            if (settings.hasOwnProperty('duden_markup_element') && !settings.hasOwnProperty('markupElement')) {
                settings.markupElement = settings.duden_markup_element;
            }

            if (settings.hasOwnProperty('duden_default_text_language') && !settings.hasOwnProperty('textLanguage')) {
                settings.textLanguage = settings.duden_default_text_language;
            }

            if (settings.hasOwnProperty('duden_proxy_script_url')) {
                settings.apiEndpoint = settings.duden_proxy_script_url;
                if (!settings.hasOwnProperty('apiEndpointScheme')) {
                    settings.apiEndpointScheme = '{apiEndpoint}?function={apiMethod}';
                }
            }

            if (!settings.hasOwnProperty('apiEndpoint')) {
                settings.apiEndpoint = defaultApiEndpoint;
                settings.apiEndpointScheme = '{apiEndpoint}?function={apiMethod}';
            }

            return settings;
        }

        /**
         * Toggle the Spellchecker on/off
         */
        toggle() {
            if (this.locked) return;

            this.api.activate(!this.api.isActive(), true);
            this.storage.setActive(this.api.isActive());
        }
    }

    const enableDKCE = (dkce, enable) => {
        dkce.locked = !!!enable;
        dkce.api.activate(enable, false);
        dkce.storage.setActive(dkce.api.isActive());
    };

    function copyDudenAttributesAndClasses(el1, el2) {
        if (!el1 || !el2) return;

        for (let attr of el1.attributes) {
            if (attr.name.startsWith('duden-') || attr.name.startsWith('dkce-')) {
                el2.setAttribute(attr.name, attr.value);
            }
        }

        for (let cls of el1.classList) {
            if (cls.startsWith('duden-') || cls.startsWith('dkce-')) {
                el2.classList.add(cls);
            }
        }
    }

    if (tinymce.majorVersion === "3") {
        tinymce.create('tinymce.plugins.DudenPlugin', {
            init : function(ed, url) {
                ed.languageSelectionActive = false;
                ed.syncAutoCorrectionMenuItem = () => {};
                ed.syncLanguageSelection = () => {};

                ed.dkce = new DKCEPlugin(ed, url);

                ed.dkce.enable = (enable) => {
                    enableDKCE(ed.dkce, enable);
                }

                ed.addButton("duden_check", {
                    title: 'button.start_correction',
                    icon: false,
                    image: url + "/img/duden-icon.png",
                    onclick: () => {
                        ed.dkce.toggle();
                        ed.syncAutoCorrectionMenuItem();
                    }
                });

                ed.onInit.add((e) => {
                    copyDudenAttributesAndClasses(ed.getElement(), ed.getBody());
                    ed.dkce.init(() => {
                        ed.syncLanguageSelection();
                    });
                })
            },

            createControl: (function (controlName, controlManager) {
                var pluginUrl = controlManager.editor.dkce.pluginUrl;
                var ed = controlManager.editor;

                switch (controlName) {
                    case 'duden_menu':

                        var menuDefinition = {
                            icon: true,
                            context: "tools",
                            menu: [{
                                text: 'menu.auto_spell_check',
                                selectable: true,
                                onclick: function () { ed.dkce.toggle(); },
                                isSelected: function () { return ed.dkce.api.isActive(); }
                            },
                            { text: '-' },
                            {
                                text: "menu.dictionary_proposal",
                                icon: false,
                                onclick: function () {
                                    var dialogDefinition = {
                                        title: 'dialog.proposal.title',
                                        body: [{
                                            type: 'textbox',
                                            name: 'proposal',
                                            label: '',
                                            value: ''
                                        }],
                                        height: "100px"
                                    };

                                    var generalDialogOptions = {
                                        inline: "yes",
                                        width: "426px",
                                        height: dialogDefinition.height,
                                        url: ed.dkce.api.getCurrentSettings().path + '/html/dialog.html',
                                        body: 'custom'
                                    };

                                    // NOTE(jochen): This is required, so the window
                                    // manager uses the correct editor instance
                                    // In the UI it is possible to activate this menu
                                    // item while another TineMCE is active
                                    tinymce.EditorManager._setActive(ed);

                                    ed.windowManager.open(generalDialogOptions, {
                                        title: dialogDefinition.title,
                                        body: dialogDefinition.body,
                                        onsubmit: function (e) {
                                            var word = e.data.proposal.trim();
                                            if (word.length > 0) {
                                                ed.dkce.api.proposeWord(word);
                                            }
                                        }
                                    });
                                }
                            },
                            { text: '-' },
                            {
                                text: "menu.dictionary_settings",
                                icon: false,
                                onclick: function (a,b,c) {
                                    var createListBox = function (name, label, defaultValue, values) {
                                        var valueObjects = [];
                                        values.forEach(function (valueLabel) {
                                            valueObjects.push({
                                                value: valueLabel[0],
                                                text: valueLabel[1]
                                            });
                                        });
                                        var listboxDefinition = {
                                            type: "listbox",
                                            name: name,
                                            label: label,
                                            value: defaultValue,
                                            values: valueObjects
                                        };
                                        return listboxDefinition;
                                    }

                                    var settings = ed.dkce.api.getCurrentSettings();

                                    var dialogElements = [
                                        createListBox(
                                            "spellcheckingLevel", ed.translate("dialog.settings.spellcheck_level"), settings.spellcheckingLevel, [
                                                [1, ed.translate("dialog.settings.spellcheck_level.1")],
                                                [2, ed.translate("dialog.settings.spellcheck_level.2")],
                                                [3, ed.translate("dialog.settings.spellcheck_level.3")],

                                            ]),
                                        createListBox(
                                            "orthographyStandard", ed.translate("dialog.settings.orthography_standard"), settings.orthographyStandard , [
                                                ["duden", ed.translate("dialog.settings.orthography_standard.duden")],
                                                ["conservative", ed.translate("dialog.settings.orthography_standard.conservative")],
                                                ["progressive", ed.translate("dialog.settings.orthography_standard.progressive")],
                                                ["extended", ed.translate("dialog.settings.orthography_standard.extended")],
                                                ["press", ed.translate("dialog.settings.orthography_standard.press")],
                                            ])
                                    ];

                                    if (!ed.languageSelectionActive) {
                                        dialogElements.unshift(
                                            createListBox(
                                                "textLanguage", ed.translate("language_selection.title"), settings.textLanguage, [
                                                    ["de-DE", ed.translate("language_selection.de-DE")],
                                                    ["de-AT", ed.translate("language_selection.de-AT")],
                                                    ["de-CH", ed.translate("language_selection.de-CH")],
                                                ]
                                            )
                                        )
                                    }

                                    var dialogDefinition = {
                                        title: 'dialog.settings.title',
                                        body: dialogElements,
                                        height: "150px"
                                    };

                                    var generalDialogOptions = {
                                        inline: "yes",
                                        width: "426px",
                                        height: dialogDefinition.height,
                                        url: ed.dkce.api.getCurrentSettings().path + '/html/dialog.html',
                                        body: 'custom'
                                    };

                                    // NOTE(jochen): This is required, so the window
                                    // manager uses the correct editor instance
                                    // In the UI it is possible to activate this menu
                                    // item while another TineMCE is active
                                    tinymce.EditorManager._setActive(ed);

                                    ed.windowManager.open(generalDialogOptions, {
                                        title: dialogDefinition.title,
                                        body: dialogDefinition.body,
                                        onsubmit: function (e) {
                                            ed.dkce.api.updateSettings(e.data);
                                            var settings = ed.dkce.api.getCurrentSettings();
                                            ed.dkce.storage.setOrthographyStandard(settings.orthographyStandard);
                                            ed.dkce.storage.setSpellcheckingLevel(settings.spellcheckingLevel);
                                            ed.dkce.storage.setLanguage(settings.textLanguage);
                                        }
                                    });
                                }
                            }]
                        };

                        menuDefinition.image = pluginUrl + "/img/settings-icon.png";
                        menuDefinition.text = "menu.title";

                        var menu = controlManager.createMenuButton("duden_menu", {
                            title: menuDefinition.text,
                            image: menuDefinition.image,
                            icon: true
                        });

                        menu.onRenderMenu.add(function (c, menu) {
                            menuDefinition.menu.forEach(function (menuItemDefinition) {
                                if (menuItemDefinition.text === "-") {
                                    menu.addSeparator();
                                } else {
                                    var menuItem;
                                    if (menuItemDefinition.selectable) {
                                        var originalOnclick = menuItemDefinition.onclick;
                                        menuItemDefinition.onclick = function () {
                                            originalOnclick();
                                            menuItem.setSelected(menuItemDefinition.isSelected())
                                        }

                                    }

                                    menuItem = menu.add({
                                        title: menuItemDefinition.text,
                                        icon: menuItemDefinition.icon,
                                        onclick: menuItemDefinition.onclick
                                    });

                                    if (menuItemDefinition.selectable) {
                                        ed.syncAutoCorrectionMenuItem = () => {
                                            menuItem.setSelected(menuItemDefinition.isSelected());
                                        }

                                        ed.syncAutoCorrectionMenuItem();
                                    }
                                }
                            });
                        });

                        // Return the new menu button instance
                        return menu;

                    case 'duden_lang':
                        ed.languageSelectionActive = true;
                        var languageListBox = controlManager.createListBox("duden_language_selection", {
                            title: 'language_selection.title',
                            onselect: function (languageCode) {
                                ed.dkce.storage.setLanguage(languageCode);
                                ed.dkce.api.updateSettings({
                                    'textLanguage': languageCode
                                });
                            }
                        });

                        ["de-DE", "de-AT", "de-CH"].forEach(function (languageCode) {
                            var languageName = controlManager.editor.translate('language_selection.' + languageCode);
                            languageListBox.add(languageName, languageCode);
                        });

                        controlManager.editor.onInit.add(function () {
                            document.getElementById(languageListBox.id + '_text').setAttribute('style', ';width:120px;');
                        });

                        ed.syncLanguageSelection = () => {
                            var currentLanguage = ed.dkce.api.getCurrentSettings().textLanguage;
                            languageListBox.select(currentLanguage);
                        }

                        return languageListBox;
                }
            }),

            getInfo: function () {
                // TODO(jochen): Check if we could optain that information from another place where
                // it's easier to maintain
                return {
                    longname : 'Duden TinyMCE Plugin',
                    author : 'EPC Consulting und Software GmbH',
                    authorurl : 'https://www.epc.de',
                    infourl : 'https://www.epc.de/aktuelles/2016/impressum',
                    version : "1.0"
                };
            }
        });

        tinymce.PluginManager.add('duden', tinymce.plugins.DudenPlugin);
    } else if (tinymce.majorVersion === "4") {
        tinymce.create('tinymce.plugins.DudenPlugin', {
            init: function (ed, url) {
                ed.dkce = new DKCEPlugin(ed, url);

                ed.dkce.dudenButton = null;

                ed.dkce.enable = (enable) => {
                    enableDKCE(ed.dkce, enable);
                    ed.dkce.dudenButton.disabled(!!!enable);
                }

                ed.on('init', function (plugin, a) {
                    copyDudenAttributesAndClasses(ed.getElement(), ed.getBody());
                    ed.dkce.init();
                });

                ed.addButton('duden_check', {
                    icon: false,
                    image: url + "/img/duden-icon.png",
                    onclick: function (api) {
                        ed.dkce.toggle();
                        this.active(ed.dkce.storage.getActive());
                    },
                    onpostrender: function () {
                        ed.dkce.dudenButton = this;
                        this.active(ed.dkce.storage.getActive());
                    }
                });
            },
        });

        tinymce.PluginManager.add('duden', tinymce.plugins.DudenPlugin);
    } else if (tinymce.majorVersion === "5" || tinymce.majorVersion === "6" || tinymce.majorVersion === "7") {
        tinymce.PluginManager.add("duden", function (ed, url) {
            ed.dkce = new DKCEPlugin(ed, url);

            // NOTE(jochen): We're patching the TinyMCE setEditableRoot method
            // so we don't need to register an obsever to detect changes of the
            // editable state
            const originalSetEditableRoot = ed.setEditableRoot;
            ed.setEditableRoot = (editableRoot) => {
                const result = originalSetEditableRoot.call(ed, editableRoot);
                ed.dkce.api.reinitialize();
                return result;
            };

            ed.dkce.enable = (enable) => {
                enableDKCE(ed.dkce, enable);
                ed.updateDudenIcon();
            }

            ed.updateDudenIcon = () => {
                var label = ed.editorManager.i18n.translate('button.start_correction');

                var element = null;

                if (ed.hasOwnProperty('editorContainer')) {
                    element = ed.editorContainer.querySelector('[aria-label="' + label + '"]');
                }

                if (!element) {
                    element = document.querySelector('[aria-label="' + label + '"]');
                }

                if (element) {
                    if (ed.dkce.storage.getActive()) {
                        element.style.filter = '';
                        element.style.opacity = '';
                    } else {
                        element.style.filter = 'grayscale(100%)';
                        element.style.opacity = 0.5;
                    }
                }
            };

            ed.ui.registry.addIcon('duden', '<svg xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" id="body_1" width="32" height="32"><g transform="matrix(0.04188482 0 0 0.04188482 0 0)"><path d="M0 0L764 0L764 764L0 764L0 0zM132 58L132 707L632 707L632 58L132 58z" stroke="#000000" stroke-width="1" fill="none" /><path d="M136.5 60L627.5 60L630 62.5L630 586.5L628.5 588L628 63.5L626.5 62L137.5 62L136 63.5L135.5 588L134 586.5L134 62.5L136.5 60z" stroke="#000000" stroke-width="1" fill="#000000" fill-rule="nonzero" fill-opacity="0.87058824" /><path d="M135.5 652L136 701.5L137.5 703L626.5 703L628 701.5L628.5 652L630 653.5L630 702.5L627.5 705L136.5 705L134 702.5L134 653.5L135.5 652z" stroke="#000000" stroke-width="1" fill="#000000" fill-rule="nonzero" fill-opacity="0.87058824" /><path d="M308.5 134L325 134.5L308.5 135L308.5 134z" stroke="#F4F4F4" stroke-width="1" fill="#F4F4F4" fill-rule="nonzero" /><path d="M337.5 134L344 134.5L337.5 135L337.5 134z" stroke="#F4F4F4" stroke-width="1" fill="#F4F4F4" fill-rule="nonzero" /><path d="M285.5 135L289 135.5L285.5 136L285.5 135z" stroke="#F4F4F4" stroke-width="1" fill="#F4F4F4" fill-rule="nonzero" /><path d="M272.5 136L276 136.5L272.5 137L272.5 136z" stroke="#F4F4F4" stroke-width="1" fill="#F4F4F4" fill-rule="nonzero" /><path d="M261.5 137L264 137.5L261.5 138L261.5 137z" stroke="#F4F4F4" stroke-width="1" fill="#F4F4F4" fill-rule="nonzero" /><path d="M303.5 138L347.5 138L348.5 139L364.5 139L365.5 140L387.5 142L416.5 149C 453.43335 160.40001 483.26666 178.90001 506 204.5C 523.4 223.43333 536.4 246.76666 545 274.5L545 274.5L551 302.5L551 311.5L552 312.5L552 343.5L551 344.5L550 361.5L545 385.5L532 420.5C 521.5333 441.7 508.03336 459.86667 491.5 475C 472.1667 493.00003 448.5 506.6667 420.5 516L420.5 516L393.5 523L375.5 525L374.5 526L348.5 527L347.5 528L321.5 528L320.5 527L297.5 527L296.5 526L283.5 526L282.5 525L238.5 522L232 519.5L232 145.5L234.5 145L250.5 142L282.5 140L283.5 139L302.5 139L303.5 138zM307 139L306 140L285 140L284 141L261 142L260 143L237 145L236 146L233 147L233 520L240 521L250 521L251 522L261 522L262 523L272 523L273 524L284 524L285 525L325 526L326 527L366 526L375 524L382 524L410 518L431 511C 455 501.00003 476 487.6667 494 471C 510.6667 454.33334 524.3334 434.33334 535 411L535 411L546 377L550 354L550 344L551 343L550 304L546 283L540 264C 531.3334 240 519 220 503 204C 479.6667 178.66667 449.6667 160.33334 413 149L413 149L387 143L366 141L365 140L307 139z" stroke="#F4F4F4" stroke-width="1" fill="#F4F4F4" fill-rule="nonzero" /><path d="M475.5 173L476.5 175L475.5 173z" stroke="#F4F4F4" stroke-width="1" fill="#F4F4F4" fill-rule="nonzero" /><path d="M478.5 175L479.5 177L478.5 175z" stroke="#F4F4F4" stroke-width="1" fill="#F4F4F4" fill-rule="nonzero" /><path d="M482.5 178L483.5 180L482.5 178z" stroke="#F4F4F4" stroke-width="1" fill="#F4F4F4" fill-rule="nonzero" /><path d="M308.5 179L358.5 179L359.5 180L375.5 181L402.5 187L423.5 195L438.5 203C 450.5667 210.26668 461.0667 219.1 470 229.5C 482.93335 243.90001 492.26666 261.90002 498 283.5L498 283.5L503 310.5L503 349.5L502 350.5L500 369.5L490 402.5C 481.4667 421.96667 469.63336 438.13336 454.5 451C 439.76666 463.6 421.76666 472.93335 400.5 479L400.5 479L368.5 485L326.5 486L325.5 485L288.5 484L283.5 483L280 480.5L279 475.5L279 188.5L280 184.5L282.5 182L293.5 180L307.5 180L308.5 179zM310 180L309 181L295 181L285 183L281 187L280 191L280 474L282 480L290 483L329 484L330 485L368 484L400 478C 421.33334 472.00003 439.6667 462.33337 455 449C 469 436.33337 480.33334 420.6667 489 402L489 402L499 369L501 350L502 349L502 312L497 285C 491 262.33334 481 243.66669 467 229C 457.6667 219 447.33334 210.33334 436 203L436 203L405 189L381 183L369 182L368 181L359 181L358 180L310 180z" stroke="#F4F4F4" stroke-width="1" fill="#F4F4F4" fill-rule="nonzero" /><path d="M486.5 181L488.5 184L486.5 181z" stroke="#F4F4F4" stroke-width="1" fill="#F4F4F4" fill-rule="nonzero" /><path d="M322.5 183L350 183.5L322.5 184L322.5 183z" stroke="#F4F4F4" stroke-width="1" fill="#F4F4F4" fill-rule="nonzero" /><path d="M302.5 184L306 184.5L302.5 185L302.5 184z" stroke="#F4F4F4" stroke-width="1" fill="#F4F4F4" fill-rule="nonzero" /><path d="M360.5 184L363 184.5L360.5 185L360.5 184z" stroke="#F4F4F4" stroke-width="1" fill="#F4F4F4" fill-rule="nonzero" /><path d="M289.5 185L292 185.5L289.5 186L289.5 185z" stroke="#F4F4F4" stroke-width="1" fill="#F4F4F4" fill-rule="nonzero" /><path d="M369.5 185L372 185.5L369.5 186L369.5 185z" stroke="#F4F4F4" stroke-width="1" fill="#F4F4F4" fill-rule="nonzero" /><path d="M492.5 186L495.5 190L492.5 186z" stroke="#F4F4F4" stroke-width="1" fill="#F4F4F4" fill-rule="nonzero" /><path d="M506.5 200L510.5 205L506.5 200z" stroke="#F4F4F4" stroke-width="1" fill="#F4F4F4" fill-rule="nonzero" /><path d="M514.5 209L515.5 211L514.5 209z" stroke="#F4F4F4" stroke-width="1" fill="#F4F4F4" fill-rule="nonzero" /><path d="M518.5 214L519.5 216L518.5 214z" stroke="#F4F4F4" stroke-width="1" fill="#F4F4F4" fill-rule="nonzero" /><path d="M447.5 215L449.5 218L447.5 215z" stroke="#F4F4F4" stroke-width="1" fill="#F4F4F4" fill-rule="nonzero" /><path d="M452.5 219L466.5 234L452.5 219z" stroke="#F4F4F4" stroke-width="1" fill="#F4F4F4" fill-rule="nonzero" /><path d="M528.5 229L529.5 231L528.5 229z" stroke="#F4F4F4" stroke-width="1" fill="#F4F4F4" fill-rule="nonzero" /><path d="M469.5 237L470.5 239L469.5 237z" stroke="#F4F4F4" stroke-width="1" fill="#F4F4F4" fill-rule="nonzero" /><path d="M472.5 241L473.5 243L472.5 241z" stroke="#F4F4F4" stroke-width="1" fill="#F4F4F4" fill-rule="nonzero" /><path d="M474.5 244L475.5 246L474.5 244z" stroke="#F4F4F4" stroke-width="1" fill="#F4F4F4" fill-rule="nonzero" /><path d="M476.5 247L477.5 249L476.5 247z" stroke="#F4F4F4" stroke-width="1" fill="#F4F4F4" fill-rule="nonzero" /><path d="M554.5 303L555 305.5L554 305.5L554.5 303z" stroke="#F4F4F4" stroke-width="1" fill="#F4F4F4" fill-rule="nonzero" /><path d="M555.5 314L556 318.5L555 318.5L555.5 314z" stroke="#F4F4F4" stroke-width="1" fill="#F4F4F4" fill-rule="nonzero" /><path d="M498.5 318L499 340.5L498 340.5L498.5 318z" stroke="#F4F4F4" stroke-width="1" fill="#F4F4F4" fill-rule="nonzero" /><path d="M555.5 336L556 341.5L555 341.5L555.5 336z" stroke="#F4F4F4" stroke-width="1" fill="#F4F4F4" fill-rule="nonzero" /><path d="M497.5 350L498 352.5L497 352.5L497.5 350z" stroke="#F4F4F4" stroke-width="1" fill="#F4F4F4" fill-rule="nonzero" /><path d="M554.5 351L555 353.5L554 353.5L554.5 351z" stroke="#F4F4F4" stroke-width="1" fill="#F4F4F4" fill-rule="nonzero" /><path d="M469.5 427L467.5 430L469.5 427z" stroke="#F4F4F4" stroke-width="1" fill="#F4F4F4" fill-rule="nonzero" /><path d="M465.5 432L461.5 437L465.5 432z" stroke="#F4F4F4" stroke-width="1" fill="#F4F4F4" fill-rule="nonzero" /><path d="M526.5 437L525.5 439L526.5 437z" stroke="#F4F4F4" stroke-width="1" fill="#F4F4F4" fill-rule="nonzero" /><path d="M453.5 444L450.5 448L453.5 444z" stroke="#F4F4F4" stroke-width="1" fill="#F4F4F4" fill-rule="nonzero" /><path d="M447.5 449L446.5 451L447.5 449z" stroke="#F4F4F4" stroke-width="1" fill="#F4F4F4" fill-rule="nonzero" /><path d="M518.5 449L517.5 451L518.5 449z" stroke="#F4F4F4" stroke-width="1" fill="#F4F4F4" fill-rule="nonzero" /><path d="M443.5 452L442.5 454L443.5 452z" stroke="#F4F4F4" stroke-width="1" fill="#F4F4F4" fill-rule="nonzero" /><path d="M515.5 453L514.5 455L515.5 453z" stroke="#F4F4F4" stroke-width="1" fill="#F4F4F4" fill-rule="nonzero" /><path d="M512.5 457L510.5 460L512.5 457z" stroke="#F4F4F4" stroke-width="1" fill="#F4F4F4" fill-rule="nonzero" /><path d="M434.5 458L433.5 460L434.5 458z" stroke="#F4F4F4" stroke-width="1" fill="#F4F4F4" fill-rule="nonzero" /><path d="M506.5 464L501.5 470L506.5 464z" stroke="#F4F4F4" stroke-width="1" fill="#F4F4F4" fill-rule="nonzero" /><path d="M495.5 475L490.5 481L495.5 475z" stroke="#F4F4F4" stroke-width="1" fill="#F4F4F4" fill-rule="nonzero" /><path d="M295.5 479L299 479.5L295.5 480L295.5 479z" stroke="#F4F4F4" stroke-width="1" fill="#F4F4F4" fill-rule="nonzero" /><path d="M370.5 479L373 479.5L370.5 480L370.5 479z" stroke="#F4F4F4" stroke-width="1" fill="#F4F4F4" fill-rule="nonzero" /><path d="M310.5 480L322 480.5L310.5 481L310.5 480z" stroke="#F4F4F4" stroke-width="1" fill="#F4F4F4" fill-rule="nonzero" /><path d="M358.5 480L363 480.5L358.5 481L358.5 480z" stroke="#F4F4F4" stroke-width="1" fill="#F4F4F4" fill-rule="nonzero" /><path d="M486.5 483L485.5 485L486.5 483z" stroke="#F4F4F4" stroke-width="1" fill="#F4F4F4" fill-rule="nonzero" /><path d="M481.5 487L480.5 489L481.5 487z" stroke="#F4F4F4" stroke-width="1" fill="#F4F4F4" fill-rule="nonzero" /><path d="M477.5 490L476.5 492L477.5 490z" stroke="#F4F4F4" stroke-width="1" fill="#F4F4F4" fill-rule="nonzero" /><path d="M240.5 525L243 525.5L240.5 526L240.5 525z" stroke="#F4F4F4" stroke-width="1" fill="#F4F4F4" fill-rule="nonzero" /><path d="M251.5 526L254 526.5L251.5 527L251.5 526z" stroke="#F4F4F4" stroke-width="1" fill="#F4F4F4" fill-rule="nonzero" /><path d="M262.5 527L265 527.5L262.5 528L262.5 527z" stroke="#F4F4F4" stroke-width="1" fill="#F4F4F4" fill-rule="nonzero" /><path d="M273.5 528L276 528.5L273.5 529L273.5 528z" stroke="#F4F4F4" stroke-width="1" fill="#F4F4F4" fill-rule="nonzero" /><path d="M285.5 529L288 529.5L285.5 530L285.5 529z" stroke="#F4F4F4" stroke-width="1" fill="#F4F4F4" fill-rule="nonzero" /><path d="M300.5 530L304 530.5L300.5 531L300.5 530z" stroke="#F4F4F4" stroke-width="1" fill="#F4F4F4" fill-rule="nonzero" /><path d="M328.5 531L343 531.5L328.5 532L328.5 531z" stroke="#F4F4F4" stroke-width="1" fill="#F4F4F4" fill-rule="nonzero" /><path d="M132 58L632 58L632 591L132 591L132 58zM137 60L134 63L134 587L136 588L136 64L138 62L627 62L628 64L629 588L630 587L630 63L628 60L137 60zM297 134L296 135L280 135L279 136L258 137L257 138L241 139L240 140L229 140L228 141L228 524L236 526L246 526L257 528L267 528L268 529L279 529L280 530L292 530L293 531L309 531L310 532L357 532L358 531L369 531L370 530L385 529L416 522C 447.33337 512 474.00003 497 496 477C 513.3334 460.33334 527.3334 440.6667 538 418L538 418L548 391L553 370L554 359L555 358L555 350L556 349L556 309L555 308L555 301L552 285L548 270C 538.6667 242.66667 525.3334 219.66667 508 201C 497.33334 188.33334 485 177.66667 471 169C 455.6667 159.66667 439.33337 152 422 146L422 146L399 140L377 136L369 136L368 135L357 135L356 134L297 134z" stroke="#000000" stroke-width="1" fill="#000000" fill-rule="nonzero" /><path d="M316.5 184L354.5 184L355.5 185L372.5 186L396.5 191C 409.63336 194.86668 421.63336 199.86668 432.5 206C 445.03333 213.13335 455.86667 221.96667 465 232.5C 477.2 245.96667 486.2 262.63336 492 282.5L492 282.5L498 314.5L498 344.5L497 345.5L497 354.5L494 371.5L490 386.5L483 404.5C 474.86667 421.36667 464.03333 435.53333 450.5 447C 436.30005 459.13336 418.9667 468.13336 398.5 474L398.5 474L373.5 479L345.5 480L342.5 481L341.5 480L306.5 480L305.5 479L292.5 479L291.5 478L285 478L285 187.5L286.5 186L315.5 185L316.5 184z" stroke="#000000" stroke-width="1" fill="#000000" fill-rule="nonzero" /><path d="M132 649L631.5 649L632 649.5L632 707L132.5 707L132 706.5L132 649zM136 652L134 654L134 703L137 705L628 705L630 703L630 654L629 652L628 702L627 703L138 703L136 702L136 652z" stroke="#000000" stroke-width="1" fill="#000000" fill-rule="nonzero" /><path d="M325.5 134L361.5 135L362.5 136L378.5 137L399.5 141L430.5 150C 460.76666 161.40001 485.93335 177.90001 506 199.5C 522.13336 216.36667 534.80005 236.70001 544 260.5L544 260.5L549 275.5L554 298.5L555 318.5L556 319.5L556 335.5L555 336.5L555 350.5L554 351.5L554 358.5L551 375.5C 548 389.1667 544 401.83334 539 413.5C 528.4 437.56668 514.23334 458.06668 496.5 475C 474.36667 496.53336 446.36667 512.2 412.5 522L412.5 522L391.5 527L372.5 529L371.5 530L363.5 530L362.5 531L304.5 531L303.5 530L288.5 530L287.5 529L243.5 526L242.5 525L229.5 524L228 522.5L228 141.5L228.5 141L238.5 141L246.5 139L263.5 138L264.5 137L275.5 137L276.5 136L288.5 136L289.5 135L324.5 135L325.5 134zM304 138L303 139L284 139L283 140L261 141L261 142L284 141L285 140L306 140L307 139L347 139L348 140L365 140L349 139L348 138L304 138zM366 140L366 141L374 141L366 140zM375 141L375 142L381 142L388 144L392 144L382 141L375 141zM251 142L232 145L233 520L233 146L260 143L251 142zM393 144L393 145C 399 145.66667 404.6667 147 410 149L410 149L413 149C 407 146.33334 400.33334 144.66667 393 144zM414 149L414 150L433 156L460 169C 472.00003 176.33334 483.33337 184.66667 494 194C 518 216.00002 535 245.33334 545 282L545 282L546 282C 538 250.66667 524.6667 225 506 205C 483.33337 178.33334 452.6667 159.66667 414 149zM309 179L308 180L294 180L283 182L280 185L279 189L279 476L280 481L284 483L289 484L326 485L327 486L369 485L401 479C 421.6667 473 439.6667 463.6667 455 451C 469.6667 438.33334 481.33334 422.33334 490 403L490 403L500 370L502 351L503 350L503 311L498 284C 492 262 482.6667 244.00002 470 230C 450.6667 207.33334 424.33334 192 391 184L391 184L386 185C 415.33337 191.00002 439.33337 202.33334 458 219C 472 231.66667 483 247.66667 491 267L491 267L499 293L501 311L502 312L502 349L501 350L499 369L495 385C 489 405.6667 479.33334 423.33334 466 438C 452.6667 452.6667 436.33334 464.00003 417 472L417 472L392 480L376 483L354 484L353 485L303 484L302 483L290 483L284 481L280 474L280 191L281 187L287 182L309 181L310 180L358 180L359 181L375 182L376 183L381 183L369 180L360 180L359 179L309 179zM323 183L322 184L290 185L289 186C 285.6667 185.33334 284 185.66667 284 187L284 187L284 479L310 480L311 481L363 481L364 480L373 480L394 476C 416.00003 470 435.00003 460.6667 451 448C 465.6667 435.33334 477.33334 419.33334 486 400L486 400L495 371L497 354L498 353L498 342L499 341L498 308L492 280C 486.6667 261.33334 478.33337 246 467 234C 451 215.33334 430.6667 201.66669 406 193L406 193L372 185L351 184L350 183L323 183zM382 183L382 184L386 184L382 183zM547 282L546 287L548 292L549 303L550 304L550 314L551 315L551 343L550 344L550 354L549 355L550 362L551 355L551 345L552 344L552 313L551 312L551 303L549 296L549 291L547 282zM549 362L544 385L535 411C 525 432.33334 512.3334 451.33334 497 468C 472.33337 493.33334 439.6667 510.6667 399 520L399 520L399 521C 437 513 468 497.6667 492 475C 507.33334 460.33334 520.3334 443 531 423C 539 406.33334 545 388 549 368L549 368L549 362zM234 520L234 521L239 522L283 525L284 526L297 526L298 527L321 527L322 528L366 527L366 526L347 526L346 527L299 526L298 525L285 525L284 524L273 524L272 523L262 523L261 522L234 520zM394 521L382 524L367 525L367 526L383 525L398 522L394 521z" stroke="#FFFFFF" stroke-width="1" fill="#FFFFFF" fill-rule="nonzero" /><path d="M296.5 134L308 134.5L296.5 135L296.5 134z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M344.5 134L356 134.5L344.5 135L344.5 134z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M279.5 135L285 135.5L279.5 136L279.5 135z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M364.5 135L368 135.5L364.5 136L364.5 135z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M267.5 136L272 136.5L267.5 137L267.5 136z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M373.5 136L377 136.5L373.5 137L373.5 136z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M257.5 137L261 137.5L257.5 138L257.5 137z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M381.5 137L384 137.5L381.5 138L381.5 137z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M248.5 138L252 138.5L248.5 139L248.5 138z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M240.5 139L244 139.5L240.5 140L240.5 139z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M228.5 140L237 140.5L228.5 141L228.5 140z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M470.5 169L471.5 171L470.5 169z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M473.5 171L474.5 173L473.5 171z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M480.5 176L481.5 178L480.5 176z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M484.5 179L485.5 181L484.5 179z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M489.5 183L491.5 186L489.5 183z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M306.5 184L316 184.5L306.5 185L306.5 184z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M355.5 184L360 184.5L355.5 185L355.5 184z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M292.5 185L298 185.5L292.5 186L292.5 185z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M365.5 185L369 185.5L365.5 186L365.5 185z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M285.5 186L285 187.5L284.5 478L284 187.5L285.5 186z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M373.5 186L376 186.5L373.5 187L373.5 186z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M495.5 188L507.5 201L495.5 188z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M511.5 205L514.5 209L511.5 205z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M436.5 208L437.5 210L436.5 208z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M516.5 211L518.5 214L516.5 211z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M442.5 212L443.5 214L442.5 212z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M445.5 214L446.5 216L445.5 214z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M520.5 216L521.5 218L520.5 216z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M450.5 218L452.5 221L450.5 218z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M522.5 219L523.5 221L522.5 219z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M524.5 222L525.5 224L524.5 222z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M526.5 225L527.5 227L526.5 225z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M465.5 233L468.5 237L465.5 233z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M470.5 239L471.5 241L470.5 239z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M478.5 251L479.5 253L478.5 251z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M495.5 296L496 298.5L495 298.5L495.5 296z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M554.5 300L555 302.5L554 302.5L554.5 300z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M496.5 302L497 304.5L496 304.5L496.5 302z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M555.5 308L556 313.5L555 313.5L555.5 308z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M497.5 309L498 313.5L497 313.5L497.5 309z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M555.5 342L556 348.5L555 348.5L555.5 342z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M497.5 345L498 349.5L497 349.5L497.5 345z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M554.5 354L555 357.5L554 357.5L554.5 354z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M496.5 355L497 357.5L496 357.5L496.5 355z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M553.5 361L554 363.5L553 363.5L553.5 361z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M476.5 416L475.5 418L476.5 416z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M470.5 425L469.5 427L470.5 425z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M466.5 430L464.5 433L466.5 430z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M460.5 437L454.5 444L460.5 437z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M524.5 441L523.5 443L524.5 441z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M522.5 444L521.5 446L522.5 444z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M449.5 447L448.5 449L449.5 447z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M520.5 447L519.5 449L520.5 447z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M445.5 450L444.5 452L445.5 450z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M517.5 451L516.5 453L517.5 451z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M441.5 453L440.5 455L441.5 453z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M438.5 455L437.5 457L438.5 455z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M514.5 455L513.5 457L514.5 455z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M510.5 460L507.5 464L510.5 460z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M502.5 469L495.5 477L502.5 469z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M385.5 476L388 476.5L385.5 477L385.5 476z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M380.5 477L383 477.5L380.5 478L380.5 477z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M286.5 478L292 478.5L286.5 479L286.5 478z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M374.5 478L377 478.5L374.5 479L374.5 478z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M299.5 479L306 479.5L299.5 480L299.5 479z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M365.5 479L370 479.5L365.5 480L365.5 479z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M322.5 480L342 480.5L322.5 481L322.5 480z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M345.5 480L358 480.5L345.5 481L345.5 480z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M489.5 481L487.5 484L489.5 481z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M484.5 485L482.5 488L484.5 485z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M479.5 489L478.5 491L479.5 489z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M472.5 494L471.5 496L472.5 494z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M469.5 496L468.5 498L469.5 496z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M466.5 498L465.5 500L466.5 498z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M235.5 525L240 525.5L235.5 526L235.5 525z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M246.5 526L251 526.5L246.5 527L246.5 526z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M256.5 527L262 527.5L256.5 528L256.5 527z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M267.5 528L273 528.5L267.5 529L267.5 528z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M382.5 528L385 528.5L382.5 529L382.5 528z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M279.5 529L285 529.5L279.5 530L279.5 529z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M374.5 529L378 529.5L374.5 530L374.5 529z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M292.5 530L300 530.5L292.5 531L292.5 530z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M365.5 530L369 530.5L365.5 531L365.5 530z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M309.5 531L328 531.5L309.5 532L309.5 531z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M343.5 531L357 531.5L343.5 532L343.5 531z" stroke="#727272" stroke-width="1" fill="#727272" fill-rule="nonzero" /><path d="M137.5 593L626.5 593L628 594.5L629 596.5L629 643.5L626.5 647L137.5 647L136 645.5L135 643.5L135 597.5L136 597.5L136 643.5L137.5 645L626.5 645L628 643.5L628 596.5L626.5 595L137.5 595L136.5 596L136 594.5L137.5 593z" stroke="#A60B25" stroke-width="1" fill="#A60B25" fill-rule="nonzero" fill-opacity="0.99215686" /><path d="M132.5 648L632 648.5L132.5 649L132.5 648z" stroke="#A60B25" stroke-width="1" fill="#A60B25" fill-rule="nonzero" fill-opacity="0.99215686" /><path d="M132 591L631.5 591L632 591.5L632 648L132.5 648L132 647.5L132 591zM138 593L136 596L138 595L627 595L628 596L628 595L627 593L138 593zM629 596L628 644L629 644L629 596zM136 597L135 644L136 644L136 597zM137 644L136 646L138 647L627 647L628 645L627 645L138 645L137 644z" stroke="#BD0D2A" stroke-width="1" fill="#BD0D2A" fill-rule="nonzero" /></g></svg>');

            ed.ui.registry.addToggleButton('duden_check', {
                tooltip: 'button.start_correction',
                type: 'menubutton',
                icon: 'duden', /* alternative: spell-check */
                onAction: (api) => {
                    ed.dkce.toggle();
                    api.setActive(ed.dkce.storage.getActive());

                    ed.updateDudenIcon();
                },
                onSetup: (api) => {
                    api.setActive(ed.dkce.storage.getActive());
                }
            });

            var settings = ed.dkce.getPluginConfiguration();
            if (settings.ai) {
                ed.ui.registry.addMenuButton('duden_ai', {
                    text: ed.translate('ai.assistant.menu.title'),
                    icon: 'ai',
                    fetch: (callback) => {
                        ed.dkce.api?.hideContextMenu();
                        let actions = ed.dkce.api?.getAIActions() || [];
                        let menuItems = [];
                        for (var action of actions) {
                            let actionId = action.id;
                            menuItems.push({
                                type: 'menuitem',
                                text: action.name,
                                onAction: () => {
                                    const content = ed.selection.getContent();
                                    if (content) {
                                        ed.dkce.api?.executeAIActionOnRange(actionId, ed.selection.getRng());
                                    }
                                }
                            })
                        }
                        callback(menuItems);
                    },
                    onSetup: (api) => {
                        const toggleState = () => {
                            const hasSelection = !!ed.selection.getContent().trim();
                            if (typeof api.setDisabled === 'function') {
                                api.setDisabled(!hasSelection);
                            } else if (typeof api.setEnabled === 'function') {
                                api.setEnabled(hasSelection);
                            } else {
                                console.warn("Cannot disable/enable AI DropDown");
                            }
                        };

                        ed.on('SelectionChange', toggleState);
                        return () => ed.off('SelectionChange', toggleState);
                    }
                });
            }

            ed.on('init', function(plugin, a) {
                ed.dkce.init(() => {
                    copyDudenAttributesAndClasses(ed.getElement(), ed.getBody());
                    ed.updateDudenIcon();
                });
            });
        });

        return {
            name: "Duden TinyMCE Plugin",
            url: "https://www.epc.de"
        };
    } else {
        alert("Duden Plugin only supports TinyMCE 3-7 but not TinyMCE " + tinymce.majorVersion);
    }
})();
