import { Settings } from './settings.js';
import { sendCommand, getCurrentHostname, sendActivationMessage } from './utils.js';

(() => {
    chrome.runtime.onMessage.addListener((request, sender, callback) => {
        if (chrome.runtime.lastError) {
            return;
        }

        if (typeof (request['_jsObject']) !== "undefined") {
            request = request['_jsObject'];
        }
        if (typeof (request.action) !== "undefined") {
            switch (request.action) {
                case 'http_request':
                    let method = request.method ? request.method.toUpperCase() : 'GET';

                    let options = {
                        method: method,
                        headers: request.headers || {},
                        credentials: request.withCredentials ? 'include' : 'omit',
                        body: request.body || null
                    };

                    fetch(request.url, options)
                        .then(async response => {
                            const result = await response.text();
                            callback({
                                error: false,
                                result: result,
                                status: response.status
                            });
                        })
                        .catch(error => {
                            callback({
                                error: true,
                                result: '',
                                status: 0 // No status code available in a network error
                            });
                        });

                    return true;
            }
        }

        // Unknown message, so we just forward the callback to close the message port
        if (callback) callback();
    });

    const toggle = async () => {
        const hostname = await getCurrentHostname();

        Settings.load((state) => {
            if (typeof(state.activated[hostname]) === "undefined") {
                state.activated[hostname] = true;
            } else {
                state.activated[hostname] = !state.activated[hostname];
            }
            Settings.save(state);

            const activated = state.activated[hostname];

            chrome.runtime.sendMessage({
                action: "toggle-activation-checkbox",
                activate: activated
            });

            sendActivationMessage(activated);
        });
    };

    chrome.commands.onCommand.addListener((command) => {
        switch (command) {
            case "toggle-plugin":
                toggle();
                break;

            case "propose-word":
                sendCommand("propose-word");
                break;

            case "ignore-word":
                sendCommand("ignore-word");
                break;
        }
    });

    chrome.runtime.onInstalled.addListener(() => {
        chrome.tabs.create({
            url: chrome.runtime.getURL("setup.html")
        }, (tab) => {});
    });

    const setAutoscanState = async (activate, tabId) => {
        try {
            await chrome.tabs.sendMessage(tabId,{ subject: 'autoscan', activate: activate });
        } catch (error) {
        }
    }

    let lastActiveTabId = null;
    chrome.tabs.onActivated.addListener((activeInfo) => {
        setAutoscanState(true, activeInfo.tabId)
        if ( lastActiveTabId ) {
            setAutoscanState(false,lastActiveTabId)
        }
        lastActiveTabId = activeInfo.tabId;
    })
})();