export const $ = (query) => {
    return document.getElementById(query) || document.querySelector(query)
};

export const $$ = (query) => {
    return document.querySelectorAll(query)
};

export const parseURL = (url) => {
    const urlObj = new URL(url);
    const searchObject = {};
    const queries = urlObj.search.replace(/^\?/, '').split('&');
    queries.forEach(query => {
        const [key, value] = query.split('=');
        searchObject[key] = value;
    });

    return {
        protocol: urlObj.protocol,
        host: urlObj.host,
        hostname: urlObj.hostname,
        port: urlObj.port,
        pathname: urlObj.pathname,
        search: urlObj.search,
        searchObject: searchObject,
        hash: urlObj.hash,
        username: urlObj.username ?? '',
        password: urlObj.password ?? '',
        apiKey: urlObj.apiKey ?? ''
    };
};

export const stripTrailingSlashes = (url) => {
    while (url.endsWith('/')) {
        url = url.slice(0, -1);
    }
    return url;
}

export const fixDKSUrl = (url) => {
    url = stripTrailingSlashes(url);
    if (!url.endsWith("/api")) {
        url = url + "/api";
    }

    return url;
}

export const request = async (method, url, credentials, onSuccess, onError, timeout) => {
  if (typeof(timeout) === 'undefined') {
    timeout = 0;
  }

  const options = {
    method: method,
    headers: {},
    credentials: 'omit'
  };

  if (credentials) {
    if (credentials.hasOwnProperty('apiKey') && credentials.apiKey !== '') {
        options.headers.Authorization = `Bearer ${credentials.apiKey}`;
    } else if (credentials.hasOwnProperty('username') && credentials.hasOwnProperty('password') && credentials.username !== '' && credentials.password !== '') {
        options.headers.Authorization = `Basic ${btoa(credentials.username + ":" + credentials.password)}`;
    }
  }

  let timeoutId = null;
  if (timeout > 0) {
    const controller = new AbortController();
    options.signal = controller.signal;
    timeoutId = setTimeout(() => {
        controller.abort();
    }, timeout);
  }

  try {
    const response = await fetch(url, options);
    if (timeoutId) clearTimeout(timeoutId); // Clear timeout if request succeeds

    if (response.ok) {
      const responseText = await response.text();
      onSuccess(responseText);
    } else {
      if (onError) {
        onError(response);
      }
    }
  } catch (e) {
    if (timeoutId) clearTimeout(timeoutId); // Clear timeout if request succeeds

    if (onError) {
      if (e.name === 'AbortError') {
        e.statusText = 'Timeout';
      } else {
        e.statusText = 'Invalid URL';
      }
      e.status = '0';
      onError(e);
    }
  }
};

export const json_request = (method, url, credentials, onSuccess, onError, timeout) => {
    request(method, url, credentials, (value) => {
        try {
            const result = JSON.parse(value);
            onSuccess(result);
        } catch (error) {
            onError(error);
        }
    }, onError, timeout);
};

export const selectByValue = (elementId, value) => {
    const options = $('#' + elementId).options;
    if (typeof (options) === 'undefined') return false;
    for (let i = 0, optionsLength = options.length; i < optionsLength; i++) {
        if (options[i].value === String(value)) {
            $('#' + elementId).selectedIndex = i;
            return true;
        }
    }
    return false;
}

export class INI {
    static async load(uri) {
        return new Promise((resolve, reject) => {
            request('GET', uri, null, (value) => {
                resolve(this.parse(value));
            }, (error) => {
                console.error(error);
                resolve(null);
            })
        });
    }

    static parse(iniString) {
        const iniMap = {};
        const lines = iniString.split('\n');
        let currentSection = '';

        for (let i = 0; i < lines.length; ++i) {
            const line = lines[i].trim();
            if (line.startsWith(';') || line === '') {
                continue;
            } else if (line.startsWith('[')) {
                if (!line.endsWith(']')) {
                    console.error("Section must end with bracket in line " + i + ": " + line);
                    continue;
                }
                currentSection = line.substring(1, line.length - 1);
            } else if (line.indexOf('=') > -1) {
                const pos = line.indexOf('=');
                const key = line.substring(0, pos - 1).trim();
                const value = line.substring(pos + 1, line.length).trim();

                if ((value.startsWith('"') && value.endsWith('"')) || (value.startsWith("'") && value.endsWith("'"))) {
                    value = value.substring(1, value.length);
                }
                iniMap[currentSection === '' ? key : currentSection + '.' + key] = value.replace(/\\n/g, '\n').replace(/\\t/g, '\t');
            }
        }
        return iniMap;
    }
}

export class Localization {
    static language = {};

    static async load(languageCode, fallbackLanguage) {
        languageCode = languageCode.toLowerCase();
        Localization.language = await INI.load(chrome.runtime.getURL('assets/lang/' + languageCode + '.ini'));
        if (!Localization.language) {
            console.error("Could not load langauge: " + languageCode + ". Trying to load fallback language.");
            Localization.language = await INI.load(chrome.runtime.getURL('assets/lang/' + fallbackLanguage + '.ini'));
            if (!Localization.language) {
                console.error("Could not load fallback language: " + fallbackLanguage);
            }
        }
    }

    static updateDocument(document) {
        let elements = document.querySelectorAll('[alt]');
        elements.forEach((element) => {
            element.alt = loca(element.alt);
        })

        elements = document.querySelectorAll('[data-lang]');
        elements.forEach((element) => {
            element.innerHTML = loca(element.attributes['data-lang'].value);
        })
    }
}

export const loca = (locaKey) => {
    if (typeof (Localization.language[locaKey]) !== 'undefined') {
        return Localization.language[locaKey];
    }
    console.info("Couldn't find loca key: " + locaKey + " in language file.");
    return '{' + locaKey + '}';
}

// TODO(jochen): Dont modifty prototype
String.prototype.assign = function (search, replace) {
    return this.split(search).join(replace);
}

export const getRadioNodes = (name) => {
    const elements = document.querySelectorAll('input[name = "' + name + '"]');
    if (!elements) return [];
    return elements;
}

export const getRadioVal = (name) => {
    let val = undefined;
    const radios = getRadioNodes(name);
    for (let i = 0, len = radios.length; i < len; i++) {
        if (radios[i].checked) {
            val = radios[i].value;
            break;
        }
    }

    return val;
}

export const sendCommand = (command) => {
    chrome.tabs.query({ active: true, lastFocusedWindow: true }, (tabs) => {
        if (tabs[0]) {
            chrome.tabs.sendMessage(
                tabs[0].id,
                { subject: command },
                (response) => { }
            );
        }
    });
}

export const sendActivationMessage = (activate) => {
    chrome.tabs.query({ active: true, lastFocusedWindow: true }, (tabs) => {
        if (tabs[0]) {
            chrome.tabs.sendMessage(
                tabs[0].id,
                { subject: 'activate', activate: activate },
                (response) => { }
            );
        }
    });
}

export const setAutoscanState = (activate, tabId) => {
    chrome.tabs.sendMessage(
        tabId,
        { subject: 'autoscan', activate: activate },
        (response) => { }
    );
}

export const getCurrentHostname = async () => {
    return new Promise((resolve) => {
        chrome.tabs.query({ active: true, lastFocusedWindow: true }, (tabs) => {
            if (tabs[0]) {
                const tab = tabs[0];
                const urlInfo = parseURL(tab.url);
                return resolve(urlInfo.hostname);
            }
            return resolve(null);
        });
    });
};

export const checkDKSVersion = async (apiEndpoint, credentials, timeout) => {
    return new Promise((resolve, reject) => {
        json_request('GET', apiEndpoint + '/version', credentials, (result) => {
            resolve(result);
        }, (error) => {
            reject(error);
        }, timeout);
    });
};

export const buildUrlWithCredentials = (url, username, password) => {
    const urlInfo = parseURL(url);
    let credentials = '';
    if (username != '' && password != '') {
        credentials = username + ':' + password + '@';
    }

    return urlInfo.protocol + '//' + credentials + urlInfo.host + urlInfo.pathname;
}

export const removeCredentialsFromUrl = (url) => {
    const urlInfo = parseURL(url);
    return urlInfo.protocol + '//' + urlInfo.host + urlInfo.pathname;
}