import { $, parseURL, json_request, Localization, loca} from './utils.js';
import { Settings } from './settings.js';

(() => {
    class InfoWindow {
        constructor(defaultSettings) {
            this.settings = defaultSettings;
        }

        async init() {
            await Localization.load(navigator.language.substring(0, 2), 'de');
            Localization.updateDocument(document);

            $('title').innerHTML = loca('infoWindow.title').assign('<highlight>', '<span>').assign('</highlight>', '</span>');
            $('websiteUrl').innerHTML = loca('infoWindow.websiteUrl').assign('{url}', '<span id="current_page_url" class="light">-</span>');
            $('correctionProfile').innerHTML = loca('infoWindow.correctionProfile').assign('{correctionProfiles}', '<span id="propertySets" class="light">-</span>');
            $('server_endpoint').innerHTML = loca('infoWindow.serverInfo.endpoint').assign('{apiEndpoint}', '<span id="apiEndpoint">-</span>');
            $('server_version_container').innerHTML = loca('infoWindow.serverInfo.version').assign('{version}', '<span id="server_version" class="light">-</span>');
            $('linguistic_option').innerHTML = loca('infoWindow.serverInfo.linguisticVerions').assign('{version}', '<span id="linguistic_version" class="light">-</span>');
            $('server_ping').innerHTML = loca('infoWindow.serverInfo.ping').assign('{ping}', '<span id="ping" class="light">-</span>');

            chrome.tabs.query({ active: true, lastFocusedWindow: true }, function([tab]) {
                if (chrome.runtime.lastError) {
                    return;
                }

                if (tab) {
                    const urlInfo = parseURL(tab.url);
                    $('#current_page_url').textContent = urlInfo.hostname;
                }
            });

            $('#settings').addEventListener('click', (event) => {
                chrome.runtime.openOptionsPage();
            })

            $('version').textContent = loca('infoWindow.version').assign('{version}', chrome.runtime.getManifest().version);

            $('#label_activated').classList.add('notransition');

            window.setTimeout(() => {
                $('#label_activated').classList.remove('notransition');
            }, 1);

            $('#activated').addEventListener('click', (event) => {
                this.settings.activated[$('#current_page_url').textContent] = event.target.checked;
                Settings.save(this.settings);
                this.sendActivationMessage(event.target.checked);
            });

            Settings.load((settings) => {
                this.settings = settings;

                const server = parseURL(settings.apiEndpoint);
                $('#apiEndpoint').textContent = server.protocol + '//' + server.host;
                $('#apiEndpoint').style.color = '7f7f7f';

                if (settings.activated[$('#current_page_url').textContent]) {
                    $('#activated').checked = true;
                } else {
                    $('#activated').checked = false;
                }

                $('#propertySets').textContent = settings.propertySets;

                this.updateServerInformation();
            });

            chrome.runtime.onMessage.addListener((msg, sender, response) => {
                if (typeof(msg) !== "undefined" && typeof(msg.action) !== "undefined") {
                    if (msg.action == 'toggle-activation-checkbox') {
                        const activated = msg.activate;
                        $('#activated').checked = msg.activate;
                    }
                }
            });
        }

        sendActivationMessage(activate) {
            chrome.tabs.query({ active: true, lastFocusedWindow: true }, (tabs) => {
                if (tabs.length > 0 && tabs[0].id) {
                    chrome.tabs.sendMessage(
                        tabs[0].id,
                        { subject: 'activate', activate: activate },
                        (response) => { }
                    );
                }
            });
        }

        updateServerInformation() {
            const timestamp = Date.now();
            json_request('GET', this.settings.apiEndpoint + '/version', this.settings.apiCredentials, (result) => {
                const ping = Date.now() - timestamp;
                $('#ping').textContent = ping + 'ms';

                $('#server_version').textContent = result['Server version'];
                $('#linguistic_version').textContent = result['DPF version'];
            }, (error) => {
                $('#apiEndpoint').textContent = error.statusText + ' (' + error.status + ')';
                $('#apiEndpoint').style.color = '#f00';
            });
        }
    }

    document.addEventListener('DOMContentLoaded', () => {
        const infoWindow = new InfoWindow(Settings.state);
        infoWindow.init();
    });
})();
