import { $, parseURL, fixDKSUrl, json_request, Localization, loca, checkDKSVersion, removeCredentialsFromUrl, $$} from './utils.js';
import { Settings } from './settings.js';

let autoscan = true;
class SetupPage {
    async init() {
        await Localization.load(navigator.language.substring(0, 2), 'de');
        Localization.updateDocument(document);
        $('title').innerHTML = loca('setup.title').assign('<highlight>', '<span>').assign('</highlight>', '</span>');
        $('autoconfig').value = loca('setup.introduction.autoconfig.start');
        $('autoconfig-retry').value = loca('setup.autoscan.failure.retry');
        $('close-button').value = loca('setup.autoscan.success.close');

        $('autoconfig').addEventListener('click', (event) => { this.startAutoScan() });
        $('autoconfig-retry').addEventListener('click', (event) => { this.startAutoScan() });
        $('close-button').addEventListener('click', () => { window.close() });
        $('save-credentials').addEventListener('click', () => { this.saveCredentials() });

        $('dksprompt-authType').addEventListener('change', this.updatePromptForm.bind(this));
        this.updatePromptForm();
    }

    getAuthType() {
        return $('dksprompt-authType').value;
    }

    updatePromptForm() {
        let authType = this.getAuthType();

        for (const node of $$('.authtype-username')) {
            node.style.display = (authType === 'username') ? 'block' : 'none';
        }
        for (const node of $$('.authtype-apikey')) {
            node.style.display = (authType === 'username') ? 'none' : 'block';
        }

        this.hidePromptErrors();
    }

    success(result) {
        $('success-text-2').innerHTML = loca('setup.autoscan.success.text.2')
            .assign('{apiEndpoint}', result.apiEndpoint)
            .assign('{dpfVersion}', result.version.dpf)
            .assign('{dksVersion}', result.version.dks);

        Settings.load((state) => {
            state.apiEndpoint = result.apiEndpoint;
            state.apiCredentials = result.credentials;
            Settings.save(state, () => {
                this.showSection('autoscan-success');
            });
        });

    }

    async checkCurrentConfiguration() {
        return new Promise((resolve, reject) => {
            Settings.load(async (state) => {
                if (typeof(state.apiEndpoint) === 'string' && state.apiEndpoint != '') {
                    let server = state.apiEndpoint;
                    if (state.apiCredentials) {
                        if (state.apiCredentials.hasOwnProperty('apiKey') && state.apiCredentials.apiKey != '') {
                            server = {
                                apiEndpoint: state.apiEndpoint,
                                username: state.apiCredentials.username,
                                password: state.apiCredentials.password,
                            }
                        } else {
                            server = {
                                apiEndpoint: state.apiEndpoint,
                                apiKey: state.apiCredentials.apiKey
                            }
                        }
                    }
                    const result = await this.checkServers([server]);
                    resolve(result);
                } else {
                    resolve({ success: false });
                }
            });
        });
    }

    async startAutoScan (servers) {
        this.showSection('scanning');

        const currentConfigResult = await this.checkCurrentConfiguration();
        if (currentConfigResult && currentConfigResult.success) {
            this.success(currentConfigResult);
            return;
        }

        const result = await this.checkServers(servers);
        if (result.success) {
            this.success(result);
        } else {
            if (result.apiEndpoint) {
                this.showSection('prompt-for-credentials');
                const urlInfo = parseURL(result.apiEndpoint);
                if (result.credentials !== null) {
                    $('dksprompt-api-endpoint').value = result.apiEndpoint;
                    if (this.getAuthType() == 'username') {
                        if ($('dksprompt-username').value != '') {
                            $('prompt-for-credentials-error-invalid-credentials').style.display = 'block';
                        }
                    } else {
                        if ($('dksprompt-apiKey').value != '') {
                            $('prompt-for-credentials-error-invalid-credentials').style.display = 'block';
                        }
                    }
                } else {
                    $('dksprompt-api-endpoint').value = result.apiEndpoint;
                }
            } else {
                this.showSection('autoscan-failure');
            }
        }
    }

    async getDynamicServers() {
        return new Promise((resolve, reject) => {
            json_request('GET', Settings.dynamicServerListURL, null, (result) => {
                if (Object.prototype.toString.call(result) === '[object Array]') {
                    resolve(result);
                } else {
                    console.warn("Invalid Response reading dynamic server list. Array expected.", result);
                    resolve([]);
                }
            }, (error) => {
                console.warn("Couldn't retrieve dnymaic servers");
                resolve([]);
            });
        });
    }

    async checkServers(servers) {
        if (typeof (servers) == 'undefined' || (Object.prototype.toString.call(servers) !== '[object Array]')) {
            servers = Settings.dksServerList;
            const dynamicServers = await this.getDynamicServers();
            servers = servers.concat(dynamicServers);
        }

        for (let i = 0; i < servers.length; ++i) {
            let apiEndpoint = servers[i];
            let credentials = null;
            if (typeof(servers[i]) === 'object') {
                apiEndpoint = servers[i].apiEndpoint;
                if (servers[i].hasOwnProperty('apiKey')) {
                    credentials = {
                        apiKey:  servers[i].apiKey
                    };
                } else {
                    credentials = {
                        username: servers[i].username,
                        password: servers[i].password
                    };
                }
            }
            apiEndpoint = fixDKSUrl(apiEndpoint);

            try {
                if (!credentials) {
                    const urlInfo = parseURL(apiEndpoint);
                    if (urlInfo.hasOwnProperty('apiKey')) {
                        credentials = {
                            apiKey: urlInfo.apiKey
                        };
                        apiEndpoint = removeCredentialsFromUrl(apiEndpoint);
                    } else if (urlInfo.username !== '' || urlInfo.password != '') {
                        credentials = {
                            username: urlInfo.username,
                            password: urlInfo.password
                        };
                        apiEndpoint = removeCredentialsFromUrl(apiEndpoint);
                    }
                }
                const result = await checkDKSVersion(apiEndpoint, credentials, 4 * 1000);
                if (typeof(result) == 'object' && typeof(result['DPF version']) !== 'undefined' && typeof(result['Server version']) !== 'undefined') {
                    return {
                        success: true,
                        apiEndpoint: apiEndpoint,
                        credentials: credentials,
                        version: {
                            dpf: result['DPF version'],
                            dks: result['Server version']
                        }
                    };
                }
            } catch (httpRequest) {
                if (httpRequest.status == 401) {
                    return {
                        success: false,
                        apiEndpoint
                    };
                }
            }
        }

        return { success: false };
    }

    showSection(section) {
        const sections = [
            'intro',
            'scanning',
            'autoscan-failure',
            'autoscan-success',
            'prompt-for-credentials'
        ]
        for (let i = 0; i < sections.length; ++i) {
            $(sections[i]).style.display = (section === sections[i]) ? 'block' : 'none';
        }

    }

    hidePromptErrors() {
        $('prompt-for-credentials-error-invalid-credentials').style.display = 'none';
        $('prompt-for-credentials-error-password-match').style.display = 'none';
        $('prompt-for-credentials-error-password-missing').style.display = 'none';
        $('prompt-for-credentials-error-username-missing').style.display = 'none';
        $('prompt-for-credentials-error-apiKey-missing').style.display = 'none';
    }

    async saveCredentials() {
        this.showSection('scanning');
        const apiEndpoint = $('dksprompt-api-endpoint').value;
        const username = $('dksprompt-username').value;
        const password1 = $('dksprompt-password-1').value;
        const password2 = $('dksprompt-password-2').value;
        const apiKey = $('dksprompt-apiKey').value;

        this.hidePromptErrors();

        let prompt = false;

        let authType = this.getAuthType();

        if (authType === 'username') {
            if (username == '') {
                $('prompt-for-credentials-error-username-missing').style.display = 'block';
                prompt = true;
            }

            if (password1 == '') {
                $('prompt-for-credentials-error-password-missing').style.display = 'block';
                prompt = true;
            }

            if (password1 !== password2) {
                $('prompt-for-credentials-error-password-match').style.display = 'block';
                prompt = true;
            }
        } else {
            if (apiKey == '') {
                $('prompt-for-credentials-error-apiKey-missing').style.display = 'block';
                prompt = true;
            }
        }

        if (prompt) {
            this.showSection('prompt-for-credentials');
            return;
        }
        if (authType === 'username') {
            this.startAutoScan([{apiEndpoint: apiEndpoint, username: username, password: password1}]);
        } else {
            this.startAutoScan([{apiEndpoint: apiEndpoint, apiKey: apiKey}]);
        }
    }
};

document.addEventListener('DOMContentLoaded', () => {
    (new SetupPage()).init();
});
