import { $, parseURL, fixDKSUrl, json_request, selectByValue, Localization, loca, getRadioNodes, getRadioVal, checkDKSVersion, removeCredentialsFromUrl, $$} from './utils.js';
import { Settings } from './settings.js';

class SettingsForm {
    async init() {
        this.resetApiState();

        await Localization.load(navigator.language.substring(0, 2), 'de');
        Localization.updateDocument(document);
        $('save').value = loca('settings.save');
        $('title').innerHTML = loca('settings.title').assign('<highlight>', '<span>').assign('</highlight>', '</span>');

        this.loadSettings();

        // reload data when document gets active
        document.addEventListener("visibilitychange", () => {
            if (!document.hidden) {
                this.loadSettings();
            }
        });

        $('settings_form').addEventListener('change', () => {
            this.enableSaveButton(true);
        });

        $('settings-authType').addEventListener('change', () => {
            this.updateAuthVisibility();

            const credentials = this.getCredentials();
            if (this.areCredentialsValid(credentials)) {
                this.loadPropSets(this.getApiURL(), credentials);
            }
        });

        $('apiendpoint-use-credentials').addEventListener('click', () => {
            if ($('apiendpoint-use-credentials').checked) {
                this.enableAuthentification(true);
            } else {
                this.enableAuthentification(false);
            }

            const credentials = this.getCredentials();
            const api = this.getApiURL();
            if (!api) return;

            if (this.areCredentialsValid(credentials)) {
                this.loadPropSets(this.getApiURL(), credentials);
            }
        });

        $('apiEndpoint').addEventListener('change', () => {
            // when user entered credentials remove them and put them in username/password field
            const url = parseURL($('apiEndpoint').value);
            if (url.username != '' && url.password != '') {
                $('apiEndpoint-username').value = url.username;
                $('apiEndpoint-password').value = url.password;
                this.enableAuthentification(true);
            } else if (url.apiKey != '') {
                $('apiEndpoint-apiKey').value = url.apiKey;
            }
            $('apiEndpoint').value = removeCredentialsFromUrl($('apiEndpoint').value);

            const credentials = this.getCredentials();
            if (this.areCredentialsValid(credentials)) {
                this.loadPropSets(this.getApiURL(), credentials);
            }
        });

        for (let field of ['apiEndpoint-username', 'apiEndpoint-password', 'apiEndpoint-apiKey']) {
            $(field).addEventListener('change', () => {
                const credentials = this.getCredentials();
                if (this.areCredentialsValid(credentials)) {
                    this.loadPropSets(this.getApiURL(), credentials);
                }
            });
        }

        $('tab-basic').addEventListener('click', () => {
            $('tab-basic').classList.add('active');
            $('tab-dictionaries').classList.remove('active');

            $('settings_form').classList.remove('hidden');
            $('settings_dictionaries').classList.add('hidden');
        });

        $('tab-dictionaries').addEventListener('click', () => {
            $('tab-dictionaries').classList.add('active');
            $('tab-basic').classList.remove('active');

            $('settings_form').classList.add('hidden');
            $('settings_dictionaries').classList.remove('hidden');
        });

        // Save Settings
        $('save').addEventListener('click', () => {
            const settings = {};

            for (let key in Settings.state) {
                settings[key] = Settings.state[key];
                const el = $(key);
                if (el) {
                    if (el.type == 'checkbox') {
                        settings[key] = el.checked;
                    } else {
                        let val = el.value;
                        if (key === 'messageLevel' || key === 'spellcheckingLevel') {
                            const intVal = parseInt(val);
                            val = isNaN(intVal) ? null : intVal;
                        } else if (key === 'apiEndpoint') {
                            val = this.getApiURL();
                        }

                        if (val === '') {
                            val = null;
                        }

                        settings[key] = val;
                    }
                }
            }

            settings.apiCredentials = this.getCredentials();

            ['styleForeign', 'styleOld', 'styleRegional', 'styleColloquial', 'styleFiller'].forEach(element => {
                let val = getRadioVal(element);
                if (val === '') val = null;
                if (val === 'true') val = true;
                if (val === 'false') val = false;
                settings[element] = val;
            });

            let dictionaries = [];

            document.querySelectorAll('[name="dictionary-item"]').forEach((item) => {
                if (item.checked) {
                    dictionaries.push(item.getAttribute('data-name'));
                }
            })

            settings['dictionaries'] = dictionaries;

            Settings.save(settings, () => {
                const status = $('status');
                if (status) {
                    status.textContent = loca('settings.optionsSaved');
                    status.style.opacity = 1;
                    this.enableSaveButton(false);
                    setTimeout(() => {
                        status.style.opacity = 0;

                        // we remove the activated entry, because we won't
                        // this to update
                        settings.activated = undefined;

                        // send changes to all open tabs
                        chrome.tabs.query({}, async (tabs) => {
                            for (let i = 0; i < tabs.length; ++i) {
                                const message = { subject: 'update_settings', settings: settings };
                                try {
                                    await chrome.tabs.sendMessage(tabs[i].id, message);
                                } catch (error) {
                                }
                            }
                        });
                    }, 1500);
                }
            });
       });
    }

    areCredentialsValid(credentials) {
        if (!credentials) return false;

        const hasValidApiKey = credentials.apiKey?.trim() !== '';
        const hasValidUsernamePassword = credentials.username?.trim() !== '' && credentials.password?.trim() !== '';

        return hasValidApiKey || hasValidUsernamePassword;
    }

    enableAuthentification(enable) {
        if (enable) {
            $('credentials-form').style.maxHeight = '250px';
            $('apiendpoint-use-credentials').checked = true;
        } else {
            $('credentials-form').style.maxHeight = '45px';
            $('apiendpoint-use-credentials').checked = false;
        }
    }

    getApiURL() {
        const url = fixDKSUrl($('apiEndpoint').value);
        return url;
    }

    getCredentials() {
        if ($('apiendpoint-use-credentials').checked) {
            const username = $('apiEndpoint-username').value;
            const password = $('apiEndpoint-password').value;
            const apiKey = $('apiEndpoint-apiKey').value;
            if (this.getAuthType() === 'apiKey') {
                return { apiKey };
            } else {
                return { username, password };
            }
        };

        return null;
    }

    resetApiState() {
        this.apiState = {
            valid: false,
            dictionaries: null,
            propsets: null
        }
    }

    enableSaveButton(enabled) {
        if (enabled) {
            $('save').style.opacity = 1.0;
            $('save').style.pointerEvents = '';
        } else {
            $('save').style.opacity = 0.4;
            $('save').style.pointerEvents = 'none';
        }
    }

    updateUpdateMessage() {
        if (this.apiState.valid && this.apiState.dictionaries != null && this.apiState.propsets != null) {
            if (!this.apiState.dictionaries || !this.apiState.propsets) {
                $('update-message').classList.remove('hidden');
                return;
            }
        }

        $('update-message').classList.add('hidden');
    };

    getAuthType() {
        return $('settings-authType').value ?? 'apiKey';
    }

    updateAuthVisibility () {
        let authType = this.getAuthType();

        for (const node of $$('.authtype-username')) {
            node.style.display = (authType === 'username') ? 'block' : 'none';
        }
        for (const node of $$('.authtype-apikey')) {
            node.style.display = (authType === 'username') ? 'none' : 'block';
        }
    }

    updateUI() {
        // Update Dictionaries
        if (!Settings.dictionaries || Object.keys(Settings.dictionaries).length == 0) {
            $('dictionary_items').innerHTML = $('no_dictionaries_template').textContent;
            Localization.updateDocument($('dictionary_items'));
            return;
        }

        const tpl = $('dictionary_item_template').textContent;
        let html = '';
        let index = 0;

        for (let name in Settings.dictionaries) {
            if (Settings.dictionaries.hasOwnProperty(name)) {
                let itemHtml = tpl;
                itemHtml = itemHtml.assign('{index}', index++);
                itemHtml = itemHtml.assign('{name}', name);
                itemHtml = itemHtml.assign('{description}', Settings.dictionaries[name]);
                html += itemHtml;
            }
        }

        $('dictionary_items').innerHTML = html;

        $('dictionary_items').querySelectorAll('input').forEach((element) => {
            element.addEventListener('change', () => {
                this.enableSaveButton(true);
            })
            if (Settings.state.dictionaries.indexOf(element.getAttribute('data-name')) > -1) {
                element.checked = true;
            }
        });

        // Update Properties
        const propertySets = Settings.state.propertySets ? Settings.state.propertySets.split(',').map((item) => { return item.trim() }): [];

        const overwritteProperties = {
            'checklevel': null,
            'language': null,
            'orthstd': null,
            'stycolloquial': null,
            'styfiller': null,
            'styforeign': null,
            'styold': null,
            'styregional': null,
            'dictionaries': []
        };

        if (typeof (Settings.state.registeredPropertySets) !== "undefined") {
            for (let propSetKey in propertySets) {
                const name = propertySets[propSetKey].trim();
                const propertySet = Settings.state.registeredPropertySets[name];
                if (typeof(propertySet) !== 'undefined') {
                    if ((typeof(propertySet.enforce) !== 'undefined') && propertySet.enforce === true) {
                        if (typeof(propertySet.checklevel)      !== 'undefined') overwritteProperties['checklevel'] = propertySet.checklevel;
                        if (typeof(propertySet.language)        !== 'undefined') overwritteProperties['language'] = propertySet.language;
                        if (typeof(propertySet.orthstd)         !== 'undefined') overwritteProperties['orthstd'] = propertySet.orthstd;
                        if (typeof(propertySet.stycolloquial)   !== 'undefined') overwritteProperties['stycolloquial'] = propertySet.stycolloquial;
                        if (typeof(propertySet.styfiller)       !== 'undefined') overwritteProperties['styfiller'] = propertySet.styfiller;
                        if (typeof(propertySet.styforeign)      !== 'undefined') overwritteProperties['styforeign'] = propertySet.styforeign;
                        if (typeof(propertySet.styold)          !== 'undefined') overwritteProperties['styold'] = propertySet.styold;
                        if (typeof(propertySet.styregional)     !== 'undefined') overwritteProperties['styregional'] = propertySet.styregional;
                    }

                    if (typeof (propertySet.dictionaries) !== 'undefined') {
                        propertySet.dictionaries.forEach((item) => {
                            overwritteProperties['dictionaries'].push(item);
                        })
                    }
                }
            }
        }

        const checkElement = (id, property, defaultValue) => {
            const element = $('#' + id);
            if (property !== null) {
                element.classList.add('disabled');
                element.checked = property;
                selectByValue(id, property);
            } else {
                element.classList.remove('disabled');
                element.checked = defaultValue;
                selectByValue(id, defaultValue);
            }
        }

        const checkRadioElement = (id, property, defaultValue) => {
            let elements = getRadioNodes(id);
            for (let i = 0, len = elements.length; i < len; i++) {
                let element = elements[i];

                let radioValue = null;
                if (element.value === "true") radioValue = true;
                if (element.value === "false") radioValue = false;

                let val = property;
                if (property !== null) {
                    element.parentNode.classList.add('disabled');
                } else {
                    element.parentNode.classList.remove('disabled');
                    val = defaultValue;
                }

                if (val === radioValue) {
                    element.checked = true;
                } else {
                    element.checked = false;
                }
            }
        }

        checkElement('textLanguage', overwritteProperties.language, Settings.state.textLanguage)
        checkElement('spellcheckingLevel', overwritteProperties.checklevel, Settings.state.spellcheckingLevel)
        checkElement('orthographyStandard', overwritteProperties.orthstd, Settings.state.orthographyStandard)
        checkElement('openContextMenuOn', null, Settings.state.openContextMenuOn)
        checkRadioElement('styleColloquial', overwritteProperties.stycolloquial, Settings.state.styleColloquial);
        checkRadioElement('styleFiller', overwritteProperties.styfiller, Settings.state.styleFiller);
        checkRadioElement('styleForeign', overwritteProperties.styforeign, Settings.state.styleForeign);
        checkRadioElement('styleOld', overwritteProperties.styold, Settings.state.styleOld);
        checkRadioElement('styleRegional', overwritteProperties.styregional, Settings.state.styleRegional);

        // Update Forced Dictionaries
        overwritteProperties.dictionaries.forEach((item) => {
            $('dictionary_items').querySelectorAll('input').forEach((element) => {
                if (element.getAttribute('data-name') === item) {
                    element.checked = true;
                    element.parentNode.classList.add('disabled');
                }
            });
        })
    }

    getDictionaries(apiEndpoint, credentials) {
        json_request('GET', apiEndpoint + '/dictionaries-with-description', credentials, (result) => {
            if (typeof (result) !== "undefined") {
                this.apiState.dictionaries = true;
                this.updateUpdateMessage();
                Settings.dictionaries = result;
                this.updateUI();
                this.updateUpdateMessage();
            } else {
                console.warn("Couldn't load dictionaries");
                this.apiState.dictionaries = false;
                this.updateUpdateMessage();
            }
        }, (error) => {
            console.warn("Couldn't load dictionaries");
            Settings.dictionaries = {};
            this.updateUI();
            this.apiState.dictionaries = false;
            this.updateUpdateMessage();
        });
    }

    checkVersion(apiEndpoint, credentials, next) {
        checkDKSVersion(apiEndpoint, credentials, 4 * 1000)
            .then((result) => {
                if (typeof (result) !== "undefined") {
                    this.apiState.valid = true;
                    $('apiEndpoint').classList.remove('error');
                } else {
                    this.apiState.valid = false;
                    $('apiEndpoint').classList.add('error');
                }
                next();
            })
            .catch((error) => {
                this.apiState.valid = false;
                $('apiEndpoint').classList.add('error');
                next();
            });
    }

    loadPropSets(apiEndpoint, credentials) {
        this.resetApiState();

        this.checkVersion(apiEndpoint, credentials, () => {
            json_request('GET', apiEndpoint + '/propset', credentials, (result) => {
                Settings.state.registeredPropertySets = {};
                if (typeof (result) !== "undefined") {
                    this.apiState.propsets = true;
                    this.updateUpdateMessage();
                    for (let key in result) {
                        const propSet = result[key];
                        Settings.state.registeredPropertySets[propSet.name] = propSet;
                    }

                    this.updateUI();
                } else {
                    this.apiState.propsets = false;
                    this.updateUpdateMessage();
                    console.warn("Couldn't load property sets. Please update DKS Version");
            }
            }, (errorRequest) => {
                Settings.state.registeredPropertySets = {};
                    if (errorRequest.status == 404) {
                    this.apiState.propsets = false;
                    this.updateUpdateMessage();
                    console.warn("Couldn't load property sets. Please update DKS Version");
                }
            });

            this.getDictionaries(apiEndpoint, credentials);
        });
    }

    loadSettings() {
        Settings.load((settings) => {
            this.loadPropSets(settings.apiEndpoint, settings.apiCredentials);

            this.enableSaveButton(false);

            for (let key in settings) {
                const el = $(key);
                if (el) {
                    if (el.type == 'checkbox') {
                        el.checked = settings[key];
                    } else {
                        if (typeof (settings[key]) === 'undefined' || settings[key] === null) {
                        } else {
                            if (key === 'apiEndpoint' && settings[key].trim() != '') {

                                const url = parseURL(settings[key]);
                                if (url.username != '' && url.password != '') {
                                    settings.apiCredentials = {
                                        username: url.username,
                                        password: url.password
                                    };

                                    el.value = removeCredentialsFromUrl(settings[key]);
                                } else if (url.apiKey != '') {
                                    settings.apiCredentials = {
                                        apiKey: url.apiKey
                                    };

                                    el.value = removeCredentialsFromUrl(settings[key]);
                                } else {
                                    el.value = settings[key];
                                }
                            } else {
                                el.value = settings[key];
                            }
                        }
                    }
                }
            }

            if (settings.apiCredentials) {
                this.enableAuthentification(true);
                if (settings.apiCredentials.hasOwnProperty('apiKey') && settings.apiCredentials.apiKey != '') {
                    $('settings-authType').value = 'apiKey';
                } else {
                    $('settings-authType').value = 'username';
                }
                $('apiEndpoint-username').value = settings.apiCredentials.username ?? '';
                $('apiEndpoint-password').value = settings.apiCredentials.password ?? '';
                $('apiEndpoint-apiKey').value = settings.apiCredentials.apiKey ?? '';
                this.updateAuthVisibility();
            } else {
                this.enableAuthentification(false);
            }
        });
    }
}

document.addEventListener('DOMContentLoaded', () => {
    (new SettingsForm()).init();
});
